<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Asistencia;
use App\Models\Curso;
use App\Models\Sesion;
use App\Models\MatriculaCurso;
use App\Models\Periodo;
use App\Models\CursoDocente;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB; 
use Barryvdh\DomPDF\Facade\Pdf;
use App\Models\Instituto;
use App\Models\PlanEstudio;
use Illuminate\Support\Facades\Storage;
class AsistenciaController extends Controller
{
    public function index($curso_id)
{
    // 1. Identificar al docente autenticado
    $docente_id = Auth::user()->docente->id ?? null;

    // 2. Buscar el curso
    $curso = Curso::findOrFail($curso_id);

    // 3. Verificar que el curso esté asignado a ese docente
    $asignado = CursoDocente::where('curso_id', $curso_id)
                            ->where('docente_id', $docente_id)
                            ->exists();

    if (!$asignado) {
        abort(403, 'No tienes permiso para acceder a este curso.');
    }

    // 4. Traer las sesiones activas de ese curso
    $sesiones = Sesion::where('curso_id', $curso_id)
                      ->where('activo', 1)
                      ->orderBy('fecha', 'asc')
                      ->get();

    // 5. Buscar el periodo activo
    $periodoActivo = Periodo::where('activo', 1)->first();

    if (!$periodoActivo) {
        abort(404, 'No se encontró un periodo activo.');
    }

    // 6. Traer los estudiantes matriculados en este curso y periodo activo
    $matriculados = MatriculaCurso::with(['asistencias' => function($query) use ($sesiones) {
            $query->whereIn('sesion_id', $sesiones->pluck('id_sesion'));
        }])
        ->select(
            'matricula_cursos.id', // <- Corrección: no cambiar nombre
            DB::raw("CONCAT(postulantes.apellidos, ' ', postulantes.nombres) as nombre"),
            'postulantes.dni'
        )
        ->join('matriculas', 'matricula_cursos.matricula_id', '=', 'matriculas.id')
        ->join('cursos', 'matricula_cursos.curso_id', '=', 'cursos.id')
        ->join('postulantes', 'matriculas.postulante_id', '=', 'postulantes.id_postulante')
        ->join('periodos', 'matriculas.periodo_id', '=', 'periodos.id')
        ->where('cursos.id', $curso_id)
        ->where('periodos.activo', 1)
        ->orderBy('nombre', 'asc')
        ->get();

    // 7. Recorremos los matriculados y asignamos las asistencias por sesión
    foreach ($matriculados as $matriculado) {
        $asistenciasPorSesion = [];

        foreach ($sesiones as $sesion) {
            // Buscar si ya existe asistencia para esta sesión
            $asistencia = $matriculado->asistencias->where('sesion_id', $sesion->id_sesion)->first();

            if ($asistencia) {
                // Si existe asistencia real
                $asistenciasPorSesion[$sesion->id_sesion] = [
                    'estado' => $asistencia->estado,
                    'observaciones' => $asistencia->observaciones,
                ];
            } else {
                // Si no hay asistencia registrada
                $asistenciasPorSesion[$sesion->id_sesion] = [
                    'estado' => 'P', // Por defecto "Presente"
                    'observaciones' => null,
                ];
            }
        }

        // Asignamos la colección procesada
        $matriculado->asistencias_por_sesion = $asistenciasPorSesion;

        // (Opcional) Si quieres tener el id como matricula_curso_id:
        $matriculado->matricula_curso_id = $matriculado->id;
    }

    // 8. Devolver la vista
    return view('docentes.asistencias.index', compact('curso', 'sesiones', 'matriculados'));
}






public function store(Request $request, $curso_id)
{
    if (!$request->has('asistencias') || empty($request->asistencias)) {
        return response()->json(['success' => false, 'message' => 'No se enviaron datos de asistencia.']);
    }

    try {
        foreach ($request->asistencias as $matriculaCursoId => $sesiones) {
            foreach ($sesiones as $sesionId => $estado) {
                // Validar que la sesión y matrícula existan (opcional pero recomendado)
                if ($sesionId == 0) {
                    return response()->json(['success' => false, 'message' => 'Sesión inválida.']);
                }

                $observacion = $request->observaciones[$matriculaCursoId][$sesionId] ?? null;

                // Aquí podrías validar la existencia si quieres más robusto:
                // if (!Sesion::find($sesionId) || !MatriculaCurso::find($matriculaCursoId)) {
                //     continue; // O podrías retornar error, depende del flujo que prefieras
                // }

                Asistencia::updateOrCreate(
                    [
                        'matricula_curso_id' => $matriculaCursoId,
                        'sesion_id' => $sesionId,
                    ],
                    [
                        'estado' => $estado,
                        'observaciones' => $observacion,
                        'fecha' => date('Y-m-d'), // 👈 Agregado aquí
                    ]
                );
            }
        }

        return response()->json(['success' => true, 'message' => 'Asistencia registrada correctamente.']);
    } catch (\Exception $e) {
        \Log::error('Error registrando asistencia: '.$e->getMessage());
        return response()->json(['success' => false, 'message' => 'Error interno al guardar la asistencia.']);
    }
}


public function reportePdf($curso_id)
{
    $instituto = Instituto::first(); // Ajusta si manejas múltiples

    $docente_id = Auth::user()->docente->id ?? null;

    // Cargar relaciones existentes y válidas
    $curso = Curso::with(['docentes.usuario', 'semestre'])->findOrFail($curso_id);


    // Verificar si el docente tiene asignado este curso
    $asignado = $curso->docentes->contains('id', $docente_id);
    if (!$asignado) {
        abort(403);
    }

    // Obtener sesiones activas del curso
    $sesiones = Sesion::where('curso_id', $curso_id)
                      ->where('activo', 1)
                      ->orderBy('fecha', 'asc')
                      ->get();

    // Obtener estudiantes matriculados con sus asistencias
    $matriculados = MatriculaCurso::with(['asistencias' => function($query) use ($sesiones) {
            $query->whereIn('sesion_id', $sesiones->pluck('id_sesion'));
        }])
        ->select(
            'matricula_cursos.id',
            DB::raw("CONCAT(postulantes.apellidos, ' ', postulantes.nombres) as nombre"),
            'postulantes.dni'
        )
        ->join('matriculas', 'matricula_cursos.matricula_id', '=', 'matriculas.id')
        ->join('cursos', 'matricula_cursos.curso_id', '=', 'cursos.id')
        ->join('postulantes', 'matriculas.postulante_id', '=', 'postulantes.id_postulante')
        ->join('periodos', 'matriculas.periodo_id', '=', 'periodos.id')
        ->where('cursos.id', $curso_id)
        ->where('periodos.activo', 1)
        ->orderBy('nombre', 'asc')
        ->get();

    // Procesar asistencias por estudiante
    foreach ($matriculados as $matriculado) {
        $asistenciasPorSesion = [];
        $totalP = $totalF = $totalJ = 0;

        foreach ($sesiones as $sesion) {
            $asistencia = $matriculado->asistencias->where('sesion_id', $sesion->id_sesion)->first();
            $estado = $asistencia->estado ?? 'P';

            if ($estado === 'P') $totalP++;
            elseif ($estado === 'F') $totalF++;
            elseif ($estado === 'J') $totalJ++;

            $asistenciasPorSesion[$sesion->id_sesion] = $estado;
        }

        $matriculado->asistencias_por_sesion = $asistenciasPorSesion;
        $matriculado->totalP = $totalP;
        $matriculado->totalF = $totalF;
        $matriculado->totalJ = $totalJ;
    }

    // Obtener plan de estudio y periodo desde alguna matrícula
    $matriculaCurso = $curso->matriculaCursos()
        ->whereHas('matricula.periodo', fn($q) => $q->where('activo', 1))
        ->with('matricula.planEstudio', 'matricula.periodo', 'matricula.semestre')
        ->first();


    $planEstudio = $matriculaCurso?->matricula?->planEstudio;
    $periodo = $matriculaCurso?->matricula?->periodo;
    

    // Preparar información para el encabezado del PDF
    $docenteAsignado = $curso->docentes->firstWhere('id', $docente_id);
$curso_info = [
    'programa' => $matriculaCurso?->matricula?->planEstudio?->nombre ?? '---',
    'periodo' => $matriculaCurso?->matricula?->periodo?->nombre ?? '---',
    'semestre' => $matriculaCurso?->matricula?->semestre?->nombre ?? '---',
    'docente' => $docenteAsignado
        ? $docenteAsignado->apellido . ' ' . $docenteAsignado->nombre
        : '---',
];


    $pdf = Pdf::loadView('docentes.asistencias.reporte_pdf', [
        'curso' => $curso,
        'sesiones' => $sesiones,
        'matriculados' => $matriculados,
        'instituto' => $instituto,
        'curso_info' => $curso_info,
    ])->setPaper('A3', 'portrait');

    return $pdf->stream('reporte_asistencia_' . $curso->id . '.pdf');
}




}
    