<?php
namespace App\Http\Controllers;

use App\Models\Admision;
use App\Models\Periodo;
use App\Models\TipoAdmision;
use App\Models\Requisito;
use App\Models\TipoPago;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AdmisionController extends Controller
{
    public function index()
    {
        // Obtener todas las admisiones con sus relaciones
        $admisiones = Admision::with(['periodo', 'tipoAdmision'])->get();

        // No es necesario obtener una admisión específica en el índice
        // El código a continuación se elimina ya que está tratando de buscar una admisión específica
        // $admision = Admision::with(['requisitos', 'tiposPago'])->findOrFail($id);

        // Obtener las otras relaciones necesarias
        $periodos = Periodo::where('activo', 1)->get();
        $tiposAdmision = TipoAdmision::all();
        $requisitos = Requisito::all();
        $tiposPago = TipoPago::where('activo', 1)->get();

        // Pasar todas las variables necesarias a la vista
        return view('admisiones.index', compact('admisiones', 'periodos', 'tiposAdmision', 'requisitos', 'tiposPago'));
    }


    public function create()
    {
        $periodos = Periodo::where('activo', 1)->get();
        $tiposAdmision = TipoAdmision::where('activo', 1)->get();
        $requisitos = Requisito::where('activo', 1)->get();
        $tiposPago = TipoPago::where('activo', 1)->get();

        return view('admisiones.create', compact('periodos', 'tiposAdmision', 'requisitos', 'tiposPago'));
    }


    public function store(Request $request)
    {
        $request->validate([
            'nombre' => 'required|string|max:50',
            'id_periodo' => 'required|exists:periodos,id',
            'id_tipo_admision' => 'required|exists:tipo_admision,id_tipo_admision',
            'inicio_proceso' => 'required|date',
            'fin_proceso' => 'required|date|after_or_equal:inicio_proceso',
            'inicio_inscripciones' => 'required|date',
            'fin_inscripciones' => 'required|date|after_or_equal:inicio_inscripciones',
            'inicio_extemporaneo' => 'nullable|date',
            'fin_extemporaneo' => 'nullable|date|after_or_equal:inicio_extemporaneo',
            'fecha_examen' => 'nullable|date',
            'direccion' => 'nullable|string|max:255',
            'requisitos' => 'nullable|array',
            'tipos_pago' => 'nullable|array',
        ]);

        DB::beginTransaction();
        try {
            $admision = Admision::create($request->only([
                'nombre', 'id_periodo', 'id_tipo_admision',
                'inicio_proceso', 'fin_proceso',
                'inicio_inscripciones', 'fin_inscripciones',
                'inicio_extemporaneo', 'fin_extemporaneo',
                'fecha_examen', 'direccion',
                'activo'
            ]));

            if ($request->has('requisitos')) {
                $admision->requisitos()->sync($request->requisitos);
            }

            if ($request->has('tipos_pago')) {
                $admision->tiposPago()->sync($request->tipos_pago);
            }

            DB::commit();
            return response()->json(['success' => 'Admisión registrada correctamente.']);
        } catch (\Throwable $e) {
            DB::rollBack();
            return response()->json(['error' => 'Error al registrar la admisión.'], 500);
        }
    }

    public function edit($id)
    {
        $admision = Admision::with(['requisitos', 'tiposPago'])->findOrFail($id);
        return response()->json(['admision' => $admision]);
    }


    public function update(Request $request, $id)
    {
        $admision = Admision::findOrFail($id);

        $request->validate([
            'nombre' => 'required|string|max:50',
            'id_periodo' => 'required|exists:periodos,id',
            'id_tipo_admision' => 'required|exists:tipo_admision,id_tipo_admision',
            'inicio_proceso' => 'required|date',
            'fin_proceso' => 'required|date|after_or_equal:inicio_proceso',
            'inicio_inscripciones' => 'required|date',
            'fin_inscripciones' => 'required|date|after_or_equal:inicio_inscripciones',
            'inicio_extemporaneo' => 'nullable|date',
            'fin_extemporaneo' => 'nullable|date|after_or_equal:inicio_extemporaneo',
            'fecha_examen' => 'nullable|date',
            'direccion' => 'nullable|string|max:255',
            'requisitos' => 'nullable|array',
            'tipos_pago' => 'nullable|array',
        ]);

        DB::beginTransaction();
        try {
            $admision->update($request->only([
                'nombre', 'id_periodo', 'id_tipo_admision',
                'inicio_proceso', 'fin_proceso',
                'inicio_inscripciones', 'fin_inscripciones',
                'inicio_extemporaneo', 'fin_extemporaneo',
                'fecha_examen', 'direccion',
                'activo'
            ]));

            $admision->requisitos()->sync($request->requisitos ?? []);
            $admision->tiposPago()->sync($request->tipos_pago ?? []);

            DB::commit();
            return response()->json(['success' => 'Admisión actualizada correctamente.']);
        } catch (\Throwable $e) {
            DB::rollBack();
            return response()->json(['error' => 'Error al actualizar la admisión.'], 500);
        }
    }

    public function destroy($id)
    {
        try {
            $admision = Admision::findOrFail($id);
            $admision->requisitos()->detach();
            $admision->tiposPago()->detach();
            $admision->delete();

            return response()->json(['success' => 'Admisión eliminada correctamente.']);
        } catch (\Throwable $e) {
            return response()->json(['error' => 'No se pudo eliminar la admisión.'], 500);
        }
    }
}
