<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\PlanEstudio;
use App\Models\Periodo;
use App\Models\Curso;
use App\Models\ModuloFormativo;
use App\Models\Semestre;
use App\Models\PlanEstudioPeriodo;

class PlanesEstudioController extends Controller
{
    // Mostrar todos los planes de estudio
    public function index()
    {
        // Cargar planes de estudio con los periodos relacionados
        $periodos = Periodo::all(); // Obtén todos los periodos disponibles
        $planesEstudio = PlanEstudio::with('periodos')->get();
        return view('planes_estudio.index', compact('planesEstudio','periodos'));
    }


    // Mostrar el formulario para crear un nuevo plan de estudio
    public function create()
    {
        $periodos = Periodos::all(); // Obtén todos los periodos disponibles
        $planesEstudio = PlanEstudio::all(); // Para el select del modal
        return view('planes_estudio.create', compact('periodos', 'planesEstudio'));
    }

    // Almacenar un nuevo plan de estudio
    public function store1(Request $request)
    {
        // Validar los datos
        $request->validate([
            'nombre' => 'required|string|max:255',
            'descripcion' => 'nullable|string',
            'codigo' => 'required|string|max:50',
            'resolucion' => 'required|string|max:50',
            'tipo' => 'required|in:Modular,Transversal,Empleabilidad',
            'activo' => 'required|in:0,1',
        ]);

        // Crear el Plan de Estudio
        $planEstudio = PlanEstudio::create([
            'nombre' => $request->nombre,
            'descripcion' => $request->descripcion,
            'codigo' => $request->codigo,
            'resolucion' => $request->resolucion,
            'tipo' => $request->tipo,
            'activo' => $request->activo,
        ]);

        return response()->json([
            'message' => 'Plan de Estudio creado exitosamente',
            'planEstudio' => $planEstudio,
        ]);
    }

public function store(Request $request, \App\Services\MoodleService $moodle)
{
    // Validar los datos
    $request->validate([
        'nombre' => 'required|string|max:255',
        'descripcion' => 'nullable|string',
        'codigo' => 'required|string|max:50',
        'resolucion' => 'required|string|max:50',
        'tipo' => 'required|in:Modular,Transversal,Empleabilidad',
        'activo' => 'required|in:0,1',
    ]);

    // Crear el Plan de Estudio en Laravel
    $planEstudio = PlanEstudio::create([
        'nombre' => $request->nombre,
        'descripcion' => $request->descripcion,
        'codigo' => $request->codigo,
        'resolucion' => $request->resolucion,
        'tipo' => $request->tipo,
        'activo' => $request->activo,
    ]);

    // 🔥 Crear categoría en Moodle
    $respuesta = $moodle->crearCategoria(
        $planEstudio->nombre,
        $planEstudio->descripcion ?? '',
        0 // categoría padre
    );

    // Guardar la categoría Moodle si se creó bien
    if (isset($respuesta[0]['id'])) {
        $planEstudio->update([
            'moodle_category_id' => $respuesta[0]['id']
        ]);
    }

    return response()->json([
        'message' => 'Plan de Estudio creado exitosamente',
        'planEstudio' => $planEstudio,
        'moodle' => $respuesta
    ]);
}


    // Mostrar el formulario para editar un plan de estudio
    public function edit($id)
    {
        //
    }

    public function show(string $id)
    {
        //
    }

    // Actualizar un plan de estudio
    public function update(Request $request, $id)
    {
        // Buscar el plan de estudio por ID
        $planEstudio = PlanEstudio::findOrFail($id);

        // Validar los datos recibidos
        $request->validate([
            'nombre' => 'required|string|max:100',
            'codigo' => 'required|string|max:50',
            'resolucion' => 'nullable|string|max:50',
            'tipo' => 'required|in:Modular,transversal,empleabilidad',
            'descripcion' => 'nullable|string',
            'activo' => 'required|boolean',
        ]);

        // Actualizar los datos
        $planEstudio->update([
            'nombre' => $request->nombre,
            'codigo' => $request->codigo,
            'resolucion' => $request->resolucion,
            'tipo' => $request->tipo,
            'descripcion' => $request->descripcion,
            'activo' => $request->activo,
        ]);

        return response()->json([
            'message' => 'Plan de estudio actualizado exitosamente.'
        ]);
    }

    // Otros métodos...


    // Eliminar un plan de estudio
    public function destroy($id)
    {
        $planEstudio = PlanEstudio::findOrFail($id);
        $planEstudio->delete(); // Eliminar el plan de estudio

        // Devolver una respuesta JSON para confirmar que se eliminó correctamente
        return response()->json([
            'message' => 'Plan de estudio eliminado correctamente.'
        ]);
    }

    


    public function verMalla(Request $request)
{
    // Solo carga la lista de planes para el select
    $planesEstudio = PlanEstudio::all();

    // No filtramos nada aquí, la malla la cargará AJAX
    return view('planes_estudio.ver_malla', compact('planesEstudio'));
}


public function filtrarMalla(Request $request)
{
    $request->validate([
        'plan_estudio_id' => 'required|exists:planes_estudio,id',
    ]);

    $planEstudioId = $request->plan_estudio_id;

    $planEstudio = PlanEstudio::findOrFail($planEstudioId);

    $cursos = $planEstudio->cursos()
        ->with('semestre')
        ->orderBy('semestre_id')
        ->get();

    $cursosPorSemestre = $cursos->groupBy('semestre_id')->sortKeys();

    return view('planes_estudio.partials.malla', compact(
        'planEstudio',
        'cursosPorSemestre'
    ))->render();
}


public function obtenerPeriodos($planId)
{
    $planEstudio = PlanEstudio::with('periodos')->findOrFail($planId);
    $todosPeriodos = Periodo::all();

    return response()->json([
        'periodos' => $todosPeriodos->map(function ($periodo) use ($planEstudio) {
            return [
                'id' => $periodo->id,
                'nombre' => $periodo->nombre,
                'checked' => $planEstudio->periodos->contains($periodo->id),
            ];
        })
    ]);
}


public function asociarPeriodos(Request $request, $planId)
{
    $planEstudio = PlanEstudio::findOrFail($planId);
    $periodosSeleccionados = $request->input('periodos', []);

    $planEstudio->periodos()->sync($periodosSeleccionados);

    return redirect()->back()->with('success', 'Periodos actualizados correctamente.');
}





}
