<?php

namespace App\Http\Controllers;

use Illuminate\Support\Facades\Auth;
use App\Models\Docente;
use App\Models\Periodo;
use Illuminate\Http\Request;
use App\Models\Semestre;
use App\Models\Horario;
use App\Models\Modulo;
use App\Models\NotaFinal;
use App\Models\Postulante;
use Illuminate\Support\Facades\DB;
use App\Models\Curso;
use App\Models\Matricula;
use App\Models\Area;
use App\Models\Anuncio;

class MetricaController extends Controller
{
    public function index()
    {
        $usuario = Auth::user();
        $roles = $usuario->roles->pluck('nombre')->toArray();

        // Obtener módulos permitidos para el usuario según sus roles
        $modulosPermitidos = DB::table('modulo_roles')
            ->join('modulos', 'modulo_roles.modulo_id', '=', 'modulos.id')
            ->whereIn('rol_id', $usuario->roles->pluck('id')->toArray())
            ->pluck('modulos.nombre')
            ->toArray();

        $cursosAgrupados = [];  // NO mostrar cursos al inicio

        $cursosConNotas = []; // Para los cursos y notas de los estudiantes

        // Si el usuario tiene rol de Estudiante, cargar datos similares a antes
        if (in_array('Estudiante', $roles)) {
            $postulante = Postulante::where('usuario_id', $usuario->id)->first();
            $periodoActivo = Periodo::where('activo', 1)->first();
        
            if ($postulante && $periodoActivo) {
                $matricula = $postulante->matriculas()
                    ->with('matriculaCursos.curso')
                    ->where('periodo_id', $periodoActivo->id)
                    ->first();
        
                if ($matricula) {
                    foreach ($matricula->matriculaCursos as $matriculaCurso) {
                        $curso = $matriculaCurso->curso;
                        $nota = NotaFinal::where('estudiante_id', $postulante->id_postulante)
                            ->where('curso_id', $curso->id)
                            ->first();
                        $notaFinal = $nota ? $nota->promedio : '-';
        
                        $horario = $curso->horarios()
                            ->where('id_periodo', $periodoActivo->id)
                            ->with('docente')
                            ->first();
        
                        $docenteNombre = $horario?->docente?->nombre ?? 'No asignado';
                        $horarioTexto = $horario
                            ? $horario->dia . ' ' . substr($horario->hora_inicio, 0, 5) . '-' . substr($horario->hora_fin, 0, 5)
                            : 'No asignado';
                        $aulaTexto = $horario
                            ? ($horario->tipo_aula . ' ' . $horario->numero_aula)
                            : 'No asignada';
        
                        $cursosConNotas[] = [
                            'id' => $curso->id,
                            'nombre' => $curso->nombre,
                            'estado' => $matriculaCurso->estado,
                            'nota' => $notaFinal,
                            'docente' => $docenteNombre,
                            'horario' => $horarioTexto,
                            'aula' => $aulaTexto,
                        ];
                    }
                }
            }

            $anunciosActivos = [];

            if ($postulante && $periodoActivo) {
                $matricula = $postulante->matriculas()
                    ->with('matriculaCursos.curso', 'planEstudio') // Incluye la relación al plan
                    ->where('periodo_id', $periodoActivo->id)
                    ->first();

                if ($matricula && $matricula->plan_estudio_id) {
                    $anunciosActivos = Anuncio::where('activo', 1)
                        ->where('plan_estudio_id', $matricula->plan_estudio_id)
                        ->get();
                }
            }


        }
        // Si el usuario tiene rol de Mesa de Partes
        $tramitesMesa = [];

        if (in_array('Mesa de Partes', $roles)) {
            // Obtener el área activa asignada al usuario
            $areaId = DB::table('usuario_areas')
                ->where('usuario_id', $usuario->id)
                ->where('activo', 1)
                ->value('area_id'); // Devuelve un único area_id activo

            if ($areaId) {
                $tramitesMesa = DB::table('solicitudes_tramites as st')
                    ->join('postulantes as p', 'st.postulante_id', '=', 'p.id_postulante')
                    ->join('tramites as t', 'st.tramite_id', '=', 't.id')
                    ->whereIn('st.estado', ['pendiente', 'derivado', 'en_proceso'])
                    ->where('st.area_id', $areaId) // ahora sí, correcto
                    ->select(
                        'st.id',
                        't.nombre as tipo_tramite',
                        DB::raw("CONCAT(p.nombres, ' ', p.apellidos) as estudiante"),
                        'st.estado',
                        'st.fecha_solicitud','st.archivo'
                    )
                    ->orderByDesc('st.fecha_solicitud')
                    ->get();
            }
        }

    // 🚀 Agrega esto para enviar todas las áreas a la vista
    $areas = Area::all();

        $totalPeriodos = Periodo::count();
        $totalPostulantes = Postulante::count();
        $totalDocentes = Docente::count();
        $totalCursos = Curso::count();
        $matriculasPorPeriodo = Matricula::selectRaw('periodos.nombre as periodo, periodos.fecha_inicio, COUNT(matriculas.id) as total')
            ->join('periodos', 'matriculas.periodo_id', '=', 'periodos.id')
            ->groupBy('periodos.nombre', 'periodos.fecha_inicio')
            ->orderBy('periodos.fecha_inicio')
            ->get();

        $labels = $matriculasPorPeriodo->pluck('periodo');
        $data = $matriculasPorPeriodo->pluck('total');

        return view('dashboard', compact(
            'usuario', 'roles', 'modulosPermitidos', 'cursosAgrupados', 'cursosConNotas',
            'totalPeriodos', 'totalPostulantes', 'totalDocentes', 'totalCursos', 'labels', 'data','tramitesMesa','areas'));
    }

    public function obtenerPlanes($periodo_id)
    {
        $planes = DB::table('planes_estudio_periodos')
            ->join('planes_estudio', 'planes_estudio_periodos.plan_estudio_id', '=', 'planes_estudio.id')
            ->where('planes_estudio_periodos.periodo_id', $periodo_id)
            ->select('planes_estudio.id', 'planes_estudio.nombre')
            ->distinct()
            ->get();

        return response()->json($planes);
    }


    public function obtenerSemestres()
    {
        $semestres = DB::table('semestres')
            ->select('id', 'nombre')
            ->orderBy('nombre')
            ->get();

        return response()->json($semestres);
    }


    public function filtrarCursos(Request $request)
    {
        $usuario = Auth::user();
        $docente = Docente::where('usuario_id', $usuario->id)->first();

        if (!$docente) {
            return response()->json(['error' => 'Docente no encontrado'], 404);
        }

        $request->validate([
            'periodo_id' => 'required|integer',
            'plan_estudio_id' => 'required|integer',
            'semestre_id' => 'required|integer',
        ]);

        $horarios = Horario::with(['curso.semestre', 'curso.modulo', 'turno', 'seccion'])
            ->where('id_docente', $docente->id)
            ->where('id_periodo', $request->periodo_id)
            ->whereHas('curso', function ($q) use ($request) {
                $q->where('semestre_id', $request->semestre_id)
                ->whereHas('planesEstudio', function ($subq) use ($request) {
                    $subq->where('planes_estudio.id', $request->plan_estudio_id);
                });
            })
            ->get();

        $cursosAgrupados = $horarios->groupBy([
            fn($h) => $h->curso->semestre->nombre ?? 'Sin Semestre',
            fn($h) => $h->turno->nombre ?? 'Sin Turno',
            fn($h) => $h->seccion->nombre ?? 'Sin Sección',
        ]);

        return view('roles.partials.cursos_docente', compact('cursosAgrupados'))->render();
    }

    public function derivar(Request $request, $id)
    {
        $tramite = DB::table('solicitudes_tramites')->where('id', $id)->first();

        if (!$tramite) {
            return response()->json(['message' => 'Trámite no encontrado'], 404);
        }

        // Lógica: cambiar estado y área de destino
        DB::table('solicitudes_tramites')
            ->where('id', $id)
            ->update([
                'estado' => 'derivado',
                'area_id' => $request->input('area_id'), // Asegúrate que venga desde el frontend
                'updated_at' => now(),
            ]);

        return response()->json(['message' => 'Trámite derivado correctamente']);
    }

}
